Option Explicit

' COPY PATH TO CLIPBOARD
'  Copyright  2017 FEtki Solutions LLC (fetki.com). All rights reserved.
'  Version: 1.1
'  - Install/uninstall by simply executing the script (double-clicking the file will usually execute it).
'  - Context menu option will be installed to the registry and accessible by right-clicking on any file or folder.
'  - Use options below to modify the script behavior.
'  - Save script as a new filename and execute to add additional item to the context menu.
'---------------------------------------------------------------------------

'INSTALLATION options (should uninstall first, or save as new filename):
Const RegKey = "HKEY_CURRENT_USER"   'Alternatively "HKEY_LOCAL_MACHINE" for all users, but depends on admin rights.
Const RegName = "CopyPathToClipboard"   'Installation package name (that appears in registry)
Const AppName = "Copy Path To Clipboard"   'Name that appears in context menu
'APPLICATION Options (changes will take effect as soon as file is saved):
Const TrailingBackslash = TRUE   'If True, a backslash will be added to the end of directory paths (if missing).
Const ConvertToUNC = TRUE   'If True, drive mapping letters will be replaced with full network paths.
Const RemoveHashNames = TRUE   'If True, hash folders in the path (e.g. \#FolderName\) will be removed.
Const IncludeChevronsAlways = FALSE   'If True, path is enclosed in < and > characters (MS Outlook then converts to a link even with spaces).
Const IncludeChevronsSpaces = TRUE   'Same as above, but only when spaces exist in the path.
Const ReplaceSpacesURL = FALSE   'Replace whitespaces with %20 for use as a URL.

'*** Script start - DO NOT CHANGE ***
Dim oFS, oWS
Dim filePath

Set oWS = CreateObject("WScript.Shell")
Set oFS = CreateObject("Scripting.FileSystemObject")

If WScript.Arguments.Count = 0 Then
    If CheckInstalled Then
        If MsgBox("'" & AppName & "' is already installed." & vbCrLf & _
                  "Do you want to uninstall it? (can re-install by running script again)" _
                  , vbYesNo + vbQuestion + vbDefaultButton2, AppName) = vbYes Then
            Call UnInstall
        End If
    Else
        If MsgBox("COPY PATH TO CLIPBOARD" & vbCrLf & "Copyright  2017 FEtki Solutions LLC (fetki.com). All rights reserved." & vbCrLf & vbCrLf & _
                  "This program adds an option to the context menu that allows the filepath of a selected file/folder to be copied to the clipboard." & vbCrLf & _
                  "After installing, simply right-click on any file or folder and select '" & AppName & "'" & vbCrLf & _
                  "Program options can be adjusted by using a text editor to edit the options at the top of the script file." & vbCrLf & vbCrLf & _
                  "Click OK to install", vbOKCancel + vbQuestion + vbDefaultButton2, AppName) = vbOK Then
            Call Install
        End If
    End If
Else
        
    filePath = WScript.Arguments.Item(0)
    
    If TrailingBackslash Then
        filePath = AddTrailingBackslash(filePath)
    End If
    
    If ConvertToUNC Then
        filePath = GetUNCPath(filePath)
    End If
    
    If RemoveHashNames Then
        filePath = GetCleanHashPath(filePath)
    End If
    
    If IncludeChevronsAlways Or IncludeChevronsSpaces Then
        filePath = AddChevrons(filePath)    
    End If

    If ReplaceSpacesURL Then
        filePath = ReplaceSpaces(filePath, "%20")    
    End If

    Call PutToClipboard(filePath)
    
End If

'cleanup
Set oWS = Nothing
Set oFS = Nothing
WScript.Quit


Sub Install()
    On Error Resume Next
    Call oWS.RegWrite(RegKey & "\SOFTWARE\Classes\*\shell\" & RegName & "\", AppName, "REG_SZ")
    Call oWS.RegWrite(RegKey & "\SOFTWARE\Classes\*\shell\" & RegName & "\command\", "wscript.exe " & WScript.ScriptFullName & " ""%1""", "REG_SZ")
    Call oWS.RegWrite(RegKey & "\SOFTWARE\Classes\Folder\shell\" & RegName & "\", AppName, "REG_SZ")
    Call oWS.RegWrite(RegKey & "\SOFTWARE\Classes\Folder\shell\" & RegName & "\command\", "wscript.exe " & WScript.ScriptFullName & " ""%1""", "REG_SZ")
    If Err.Number = 0 Then
        MsgBox "Installation successful", vbInformation, AppName
    Else
        MsgBox "Error installing to registry:" & vbCrLf & Err.Description, vbCritical, AppName
    End If
    Err.Clear
    On Error GoTo 0
End Sub

Sub UnInstall()
    On Error Resume Next
    Call oWS.RegDelete(RegKey & "\SOFTWARE\Classes\*\shell\" & RegName & "\command\")
    Call oWS.RegDelete(RegKey & "\SOFTWARE\Classes\*\shell\" & RegName & "\")
    Call oWS.RegDelete(RegKey & "\SOFTWARE\Classes\Folder\shell\" & RegName & "\command\")
    Call oWS.RegDelete(RegKey & "\SOFTWARE\Classes\Folder\shell\" & RegName & "\")
    If Err.Number = 0 Then
        MsgBox "Uninstallation successful", vbInformation, AppName
    Else
        MsgBox "Error uninstalling from registry:" & vbCrLf & Err.Description, vbCritical, AppName
    End If
    Err.Clear
    On Error GoTo 0
End Sub

Sub PutToClipboard(Path)
    
    Dim blankPath
    Dim oIE
    Dim returnCode
    
    On Error Resume Next
    'try clip.exe (included with Windows 7 and on)
    'check echo first
    returnCode = oWS.Run("cmd.exe /c echo . | set /p x=""" & Path & """", 0, True)
    If returnCode = 0 Then
        'pipe path to clip.exe
        returnCode = oWS.Run("cmd.exe /c echo . | set /p x=""" & Path & """ | clip", 0, True)
    End If
    If Err.Number = 0 And returnCode > 0 Then Err.Raise 1
    
    If Err.Number <> 0 Then
        Err.Clear
        'Try IE method with local html file (avoids security popups)
        blankPath = oFS.BuildPath(oFS.GetParentFolderName(WScript.ScriptFullName), "blank.html")
        If Not oFS.FileExists(blankPath) Then
            oFS.CreateTextFile (blankPath)
            If Err.Number <> 0 Then
                Err.Clear
                'Can't use local html file, revert to default "about:blank" path
                blankPath = "about:blank"
            End If
        End If
        
        'open IE and put path in clipboard
        Set oIE = CreateObject("InternetExplorer.Application")
        Call oIE.Navigate(blankPath)
        Call oIE.Document.ParentWindow.ClipboardData.SetData("text", Path)
        oIE.Quit
        Set oIE = Nothing
        If Err.Number <> 0 Then
            MsgBox "Error copying path to clipboard:" & vbCrLf & vbCrLf & Err.Description, vbCritical, AppName
            Err.Clear
        End If
    End If
    
    On Error GoTo 0
    
End Sub

Function CheckInstalled()
    On Error Resume Next
    Call oWS.RegRead(RegKey & "\SOFTWARE\Classes\*\shell\" & RegName & "\")
    Call oWS.RegRead(RegKey & "\SOFTWARE\Classes\Folder\shell\" & RegName & "\")
    If Err.Number = 0 Then
        CheckInstalled = True
    Else
        CheckInstalled = False
        Err.Clear
    End If
    On Error GoTo 0
End Function

Function AddTrailingBackslash(Path)

    AddTrailingBackslash = Path
    If oFS.GetExtensionName(Path) = "" Then
        On Error Resume Next
        oFS.GetFolder (Path)
        If Err.Number = 0 Then
            AddTrailingBackslash = Path & "\"
        End If
        Err.Clear
        On Error GoTo 0
    End If

End Function

Function GetUNCPath(Path)
    
    Dim targetDrive, d
    
    If Left(Path, 2) = "\\" Then
        'already a UNC path
        GetUNCPath = Path
        Exit Function
    End If
    
    targetDrive = oFS.GetDriveName(Path)
    For Each d In oFS.Drives
        If d.Path = targetDrive Then
            If d.ShareName <> "" Then
                'replace drive letter with drive path
                GetUNCPath = Replace(Path, d.Path, d.ShareName)
            Else
                'drive is a local drive, return original path
                GetUNCPath = Path
            End If
            Exit Function
        End If
    Next
    'drive not found, return original path
    GetUNCPath = Path

End Function

Function GetCleanHashPath(Path)
    
    Dim pos1, pos2, tmpPath
    
    tmpPath = Path
    pos1 = InStr(1, tmpPath, "#")
    Do While pos1 > 0
        pos2 = InStr(pos1, tmpPath, "\")
        tmpPath = Left(tmpPath, pos1 - 1) & Mid(tmpPath, pos2 + 1)
        pos1 = InStr(1, tmpPath, "#")
    Loop
    
    GetCleanHashPath = tmpPath

End Function

Function AddChevrons(Path)
    
    If IncludeChevronsAlways Then
        AddChevrons = "<" & Path & ">"
    ElseIf IncludeChevronsSpaces Then
        If InStr(1, Path, " ") > 0 Then
            AddChevrons = "<" & Path & ">"
        Else
            AddChevrons = Path
        End If
    End If

End Function

Function ReplaceSpaces(Path, str)
    
    ReplaceSpaces = Replace(Path, " ", str)

End Function
